import { ApplicationTelemetry } from "../../config/ClientConfiguration";
import { Logger } from "../../logger/Logger";
import { InProgressPerformanceEvent, IPerformanceClient, PerformanceCallbackFunction } from "./IPerformanceClient";
import { IPerformanceMeasurement } from "./IPerformanceMeasurement";
import { PerformanceEvent, PerformanceEvents } from "./PerformanceEvent";
export declare abstract class PerformanceClient implements IPerformanceClient {
    protected authority: string;
    protected libraryName: string;
    protected libraryVersion: string;
    protected applicationTelemetry: ApplicationTelemetry;
    protected clientId: string;
    protected logger: Logger;
    protected callbacks: Map<string, PerformanceCallbackFunction>;
    /**
     * Multiple events with the same correlation id.
     * Double keyed by correlation id and event id.
     * @protected
     * @type {Map<string, Map<string, PerformanceEvent>>}
     */
    protected eventsByCorrelationId: Map<string, Map<string, PerformanceEvent>>;
    /**
     * Underlying performance measurements for each operation
     *
     * @protected
     * @type {Map<string, IPerformanceMeasurement>}
     */
    protected measurementsById: Map<string, IPerformanceMeasurement>;
    /**
     * Creates an instance of PerformanceClient,
     * an abstract class containing core performance telemetry logic.
     *
     * @constructor
     * @param {string} clientId Client ID of the application
     * @param {string} authority Authority used by the application
     * @param {Logger} logger Logger used by the application
     * @param {string} libraryName Name of the library
     * @param {string} libraryVersion Version of the library
     */
    constructor(clientId: string, authority: string, logger: Logger, libraryName: string, libraryVersion: string, applicationTelemetry: ApplicationTelemetry);
    /**
     * Starts and returns an platform-specific implementation of IPerformanceMeasurement.
     *
     * @abstract
     * @param {string} measureName
     * @param {string} correlationId
     * @returns {IPerformanceMeasurement}
     */
    abstract startPerformanceMeasuremeant(measureName: string, correlationId: string): IPerformanceMeasurement;
    /**
     * Generates and returns a unique id, typically a guid.
     *
     * @abstract
     * @returns {string}
     */
    abstract generateId(): string;
    /**
     * Starts measuring performance for a given operation. Returns a function that should be used to end the measurement.
     *
     * @param {PerformanceEvents} measureName
     * @param {?string} [correlationId]
     * @returns {InProgressPerformanceEvent}
     */
    startMeasurement(measureName: PerformanceEvents, correlationId?: string): InProgressPerformanceEvent;
    /**
     * Stops measuring the performance for an operation. Should only be called directly by PerformanceClient classes,
     * as consumers should instead use the function returned by startMeasurement.
     *
     * @param {PerformanceEvent} event
     * @returns {(PerformanceEvent | null)}
     */
    endMeasurement(event: PerformanceEvent): PerformanceEvent | null;
    /**
     * Upserts event into event cache.
     * First key is the correlation id, second key is the event id.
     * Allows for events to be grouped by correlation id,
     * and to easily allow for properties on them to be updated.
     *
     * @private
     * @param {PerformanceEvent} event
     */
    private cacheEventByCorrelationId;
    /**
     * Cache measurements by their id.
     *
     * @private
     * @param {PerformanceEvent} event
     * @param {IPerformanceMeasurement} measurement
     */
    private cacheMeasurement;
    /**
     * Gathers and emits performance events for measurements taked for the given top-level API and correlation ID.
     *
     * @param {PerformanceEvents} measureName
     * @param {string} correlationId
     */
    flushMeasurements(measureName: PerformanceEvents, correlationId: string): void;
    /**
     * Removes measurements for a given correlation id.
     *
     * @param {string} correlationId
     */
    discardMeasurements(correlationId: string): void;
    /**
     * Registers a callback function to receive performance events.
     *
     * @param {PerformanceCallbackFunction} callback
     * @returns {string}
     */
    addPerformanceCallback(callback: PerformanceCallbackFunction): string;
    /**
     * Removes a callback registered with addPerformanceCallback.
     *
     * @param {string} callbackId
     * @returns {boolean}
     */
    removePerformanceCallback(callbackId: string): boolean;
    /**
     * Emits events to all registered callbacks.
     *
     * @param {PerformanceEvent[]} events
     * @param {?string} [correlationId]
     */
    emitEvents(events: PerformanceEvent[], correlationId: string): void;
}
//# sourceMappingURL=PerformanceClient.d.ts.map